/**
 * Bait Alamer House Progress Tracker
 * Vanilla JavaScript Application - No external dependencies
 * 
 * DEPLOYMENT INSTRUCTIONS:
 * 1) Upload all files to cPanel public_html/
 * 2) Ensure proper file structure: /index.html, /style.css, /app.js, /data/updates.json, /assets/*, /updates/*
 * 3) Create subdomain pointing to public_html/ (optional)
 * 
 * DAILY WORKFLOW:
 * 1) Create folder: /updates/YYYY-MM-DD/
 * 2) Upload compressed photos (~1600px wide, JPG quality ~75)
 * 3) Edit /data/updates.json - add new entry with date, notes, image paths
 * 4) Upload updated updates.json
 * 5) Refresh page - newest updates appear first automatically
 * 
 * TROUBLESHOOTING:
 * - Images don't appear: check paths and case sensitivity
 * - JSON parse error: validate commas and quotes
 * - Lightbox doesn't open: check browser console for errors
 * 
 * CUSTOMIZATION:
 * - Change --accent CSS variable for brand color
 * - Replace /assets/logo.png (keep same filename)
 * - Update <title> and meta description in index.html
 */

class HouseProgressTracker {
    constructor() {
        this.data = null;
        this.scheduleData = null;
        this.filteredScheduleData = [];
        this.currentLightboxImages = [];
        this.currentImageIndex = 0;
        this.lightboxOpen = false;
        this.currentSortField = null;
        this.currentSortDirection = 'asc';
        
        // DOM elements
        this.elements = {
            loading: document.getElementById('loading'),
            error: document.getElementById('error'),
            latestContent: document.getElementById('latest-content'),
            galleryContent: document.getElementById('gallery-content'),
            timelineContent: document.getElementById('timeline-content'),
            lightbox: document.getElementById('lightbox'),
            lightboxImage: document.getElementById('lightbox-image'),
            lightboxVideo: document.getElementById('lightbox-video'),
            lightboxTitle: document.getElementById('lightbox-title'),
            lightboxCounter: document.getElementById('lightbox-counter'),
            scheduleTbody: document.getElementById('schedule-tbody'),
            scheduleSearch: document.getElementById('schedule-search'),
            logoLoading: document.getElementById('logo-loading')
        };
        
        this.init();
    }
    
    async init() {
        try {
            // Start logo animation
            this.startLogoAnimation();
            
            await this.loadData();
            await this.loadScheduleData();
            this.render();
            this.initLightbox();
            this.initNavigation();
            this.initSchedule();
        } catch (error) {
            this.showError(error);
        }
    }
    
    startLogoAnimation() {
        // Add loading class to body
        document.body.classList.add('loading');
        
        // Start the glow animation after fade-in
        setTimeout(() => {
            if (this.elements.logoLoading) {
                this.elements.logoLoading.style.animation = 'logoGlow 2s ease-in-out infinite';
            }
        }, 1000); // Start glow after 1 second (fade-in duration)
        
        // Start the exit animation after glow
        setTimeout(() => {
            if (this.elements.logoLoading) {
                this.elements.logoLoading.style.animation = 'logoExit 1s ease-out forwards';
                
                // Remove from DOM after exit animation
                setTimeout(() => {
                    if (this.elements.logoLoading) {
                        this.elements.logoLoading.style.display = 'none';
                    }
                    // Show main content
                    document.body.classList.remove('loading');
                    document.body.classList.add('loaded');
                }, 1000);
            }
        }, 3500); // Start exit after 3.5 seconds (1s fade-in + 2.5s glow)
    }
    
    async loadData() {
        try {
            const response = await fetch('data/updates.json');
            if (!response.ok) {
                throw new Error(`Failed to load updates: ${response.status} ${response.statusText}`);
            }
            this.data = await response.json();
            
            // Sort updates by date (newest first)
            this.data.updates.sort((a, b) => new Date(b.date) - new Date(a.date));
            
            this.hideLoading();
        } catch (error) {
            console.error('Error loading data:', error);
            throw new Error('Unable to load updates. Please check your connection and try again.');
        }
    }
    
    async loadScheduleData() {
        try {
            const response = await fetch('data/schedule.json');
            if (!response.ok) {
                throw new Error(`Failed to load schedule: ${response.status} ${response.statusText}`);
            }
            this.scheduleData = await response.json();
            this.filteredScheduleData = [...this.scheduleData.schedule];
        } catch (error) {
            console.error('Error loading schedule data:', error);
            // Don't throw here - schedule is optional
            this.scheduleData = { schedule: [] };
            this.filteredScheduleData = [];
        }
    }
    
    hideLoading() {
        if (this.elements.loading) {
            this.elements.loading.classList.add('hidden');
        }
    }
    
    showError(error) {
        this.hideLoading();
        if (this.elements.error) {
            this.elements.error.classList.remove('hidden');
            console.error('Application error:', error);
        }
    }
    
    render() {
        if (!this.data || !this.data.updates) {
            this.showError(new Error('No updates data available'));
            return;
        }
        
        this.renderLatest();
        this.renderGallery();
        this.renderTimeline();
        this.renderSchedule();
    }
    
    renderLatest() {
        const updates = this.data.updates;
        if (updates.length === 0) {
            this.elements.latestContent.innerHTML = '';
            return;
        }
        
        const latest = updates[0];
        const maxImages = 6; // Show first 6 images in hero section
        const displayImages = latest.images.slice(0, maxImages);
        
        this.elements.latestContent.innerHTML = `
            <div class="card">
                <div class="card-date">${this.formatDate(latest.date)}</div>
                <h3 class="card-title">Latest Progress</h3>
                ${this.renderWorkNotes(latest.notes)}
                ${this.renderImageGrid(displayImages, latest.date)}
                ${latest.images.length > maxImages ? `
                    <p style="text-align: center; margin-top: var(--space-md); color: var(--muted);">
                        <a href="#d-${latest.date}" style="color: var(--accent); text-decoration: none;">
                            View all ${latest.images.length} images →
                        </a>
                    </p>
                ` : ''}
            </div>
        `;
    }
    
    renderGallery() {
        // Check if gallery section exists (it might be commented out)
        if (!this.elements.galleryContent) {
            console.log('Gallery section is not available (commented out)');
            return;
        }
        
        const updates = this.data.updates;
        if (updates.length === 0) {
            this.elements.galleryContent.innerHTML = this.getEmptyState('No gallery images available yet.');
            return;
        }
        
        const galleryHTML = updates.map(update => `
            <div class="update-entry" id="d-${update.date}">
                <div class="update-header">
                    <h3 class="update-date">${this.formatDate(update.date)}</h3>
                    <div class="update-line"></div>
                </div>
                ${this.renderWorkNotes(update.notes)}
                ${this.renderImageGrid(update.images, update.date)}
            </div>
        `).join('');
        
        this.elements.galleryContent.innerHTML = galleryHTML;
    }
    
    renderTimeline() {
        // Use schedule data for timeline if available, otherwise fall back to updates
        if (this.scheduleData && this.scheduleData.schedule.length > 0) {
            this.renderScheduleTimeline();
        } else if (this.data && this.data.updates.length > 0) {
            this.renderUpdatesTimeline();
        } else {
            this.elements.timelineContent.innerHTML = '';
        }
    }

    renderScheduleTimeline() {
        // Filter to show only items with progress > 0, then sort by start date
        const filteredSchedule = this.scheduleData.schedule.filter(item => {
            const completionPercentage = item.completion_percentage || 0;
            return completionPercentage > 0;
        });

        const sortedSchedule = [...filteredSchedule].sort((a, b) => {
            const dateA = this.parseDate(a.start_date);
            const dateB = this.parseDate(b.start_date);
            if (!dateA && !dateB) return 0;
            if (!dateA) return 1;
            if (!dateB) return -1;
            return dateA - dateB;
        });

        // If no items have started, show empty state
        if (sortedSchedule.length === 0) {
            this.elements.timelineContent.innerHTML = `
                <div style="text-align: center; padding: var(--space-xl); color: var(--muted);">
                    <p>لم يبدأ أي عمل بعد</p>
                </div>
            `;
            return;
        }

        const timelineHTML = sortedSchedule.map((item, index) => {
            const startDate = this.parseDate(item.start_date);
            const endDate = this.parseDate(item.end_date);
            const completionPercentage = item.completion_percentage || 0;
            const statusPill = this.getSmartStatusPill(item);
            
            // Format dates for display
            const startDateText = startDate ? this.formatDate(startDate.toISOString().split('T')[0]) : item.start_date;
            const endDateText = endDate ? this.formatDate(endDate.toISOString().split('T')[0]) : item.end_date;
            
            return `
                <div class="timeline-item">
                    <div class="timeline-dot ${this.getTimelineStatus(completionPercentage)}"></div>
                    <div class="timeline-content">
                        <div class="timeline-work">${this.escapeHtml(item.work)}</div>
                        <div class="progress-bar">
                            <div class="progress-fill" style="width: ${completionPercentage}%"></div>
                            <span class="progress-text">${completionPercentage}%</span>
                        </div>
                    </div>
                </div>
            `;
        }).join('');
        
        this.elements.timelineContent.innerHTML = timelineHTML;
    }

    renderUpdatesTimeline() {
        const updates = this.data.updates;
        const timelineHTML = updates.map((update, index) => `
            <div class="timeline-item">
                <div class="timeline-dot"></div>
                <div class="timeline-date">
                    <a href="#d-${update.date}">${this.formatDate(update.date)}</a>
                </div>
                <div class="timeline-notes">
                    ${this.renderWorkNotes(update.notes, false)}
                </div>
            </div>
        `).join('');
        
        this.elements.timelineContent.innerHTML = timelineHTML;
    }

    getTimelineStatus(completionPercentage) {
        if (completionPercentage >= 100) return 'completed';
        if (completionPercentage >= 75) return 'on-track';
        if (completionPercentage >= 50) return 'in-progress';
        if (completionPercentage > 0) return 'started';
        return 'not-started';
    }
    
    renderWorkNotes(notes, inCard = true) {
        if (!notes || notes.length === 0) {
            return '<p style="color: var(--muted); font-style: italic;">No work notes for this date.</p>';
        }
        
        const listItems = notes.map(note => `<li>${this.escapeHtml(note)}</li>`).join('');
        const className = inCard ? 'work-notes' : 'timeline-notes';
        
        return `
            <div class="${className}">
                <ul>${listItems}</ul>
            </div>
        `;
    }
    
    renderImageGrid(images, date) {
        if (!images || images.length === 0) {
            return '<p style="color: var(--muted); font-style: italic; text-align: center; padding: var(--space-lg);">No images available for this date.</p>';
        }
        
        const imageItems = images.map((mediaPath, index) => {
            const isVideoFile = this.isVideo(mediaPath);
            const altText = this.generateAltText(mediaPath, date, index);
            
            if (isVideoFile) {
                return `
                    <div class="image-item video-item" 
                         data-images='${JSON.stringify(images)}' 
                         data-index="${index}"
                         data-date="${date}"
                         role="button"
                         tabindex="0"
                         aria-label="Open video ${index + 1} of ${images.length} in lightbox">
                        <video src="${this.escapeHtml(mediaPath)}" 
                               preload="metadata"
                               muted
                               playsinline
                               style="width: 100%; height: 100%; object-fit: cover; pointer-events: none;"
                               onerror="console.error('Video failed to load:', this.src)">
                        </video>
                        <div class="video-play-overlay">
                            <svg width="64" height="64" viewBox="0 0 64 64" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <circle cx="32" cy="32" r="32" fill="rgba(0, 0, 0, 0.6)"/>
                                <path d="M24 18L24 46L46 32L24 18Z" fill="white"/>
                            </svg>
                        </div>
                    </div>
                `;
            } else {
                return `
                    <div class="image-item" 
                         data-images='${JSON.stringify(images)}' 
                         data-index="${index}"
                         data-date="${date}"
                         role="button"
                         tabindex="0"
                         aria-label="Open image ${index + 1} of ${images.length} in lightbox">
                        <img src="${mediaPath}" 
                             alt="${altText}" 
                             loading="lazy"
                             width="300"
                             height="225">
                    </div>
                `;
            }
        }).join('');
        
        return `<div class="image-grid">${imageItems}</div>`;
    }
    
    getEmptyState(message) {
        return `
            <div style="text-align: center; padding: var(--space-2xl); color: var(--muted);">
                <p>${message}</p>
            </div>
        `;
    }
    
    formatDate(dateString) {
        try {
            const date = new Date(dateString);
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric' 
            };
            return date.toLocaleDateString('en-US', options);
        } catch (error) {
            console.warn('Invalid date format:', dateString);
            return dateString; // Return original string if parsing fails
        }
    }
    
    generateAltText(imagePath, date, index) {
        // Extract filename without extension for alt text
        const filename = imagePath.split('/').pop().split('.')[0];
        return `Construction progress photo ${filename} from ${this.formatDate(date)}`;
    }
    
    isVideo(filePath) {
        const videoExtensions = ['.mp4', '.webm', '.ogg', '.mov'];
        const lowerPath = filePath.toLowerCase();
        return videoExtensions.some(ext => lowerPath.endsWith(ext));
    }
    
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    initLightbox() {
        const lightbox = this.elements.lightbox;
        const backdrop = lightbox.querySelector('.lightbox-backdrop');
        const closeBtn = lightbox.querySelector('.lightbox-close');
        const prevBtn = lightbox.querySelector('.lightbox-prev');
        const nextBtn = lightbox.querySelector('.lightbox-next');
        
        // Open lightbox on image/video click
        document.addEventListener('click', (e) => {
            const imageItem = e.target.closest('.image-item');
            if (imageItem) {
                e.preventDefault();
                e.stopPropagation();
                // Pause any playing videos in the grid
                const video = imageItem.querySelector('video');
                if (video) {
                    video.pause();
                }
                this.openLightbox(imageItem);
            }
        });
        
        // Keyboard navigation for image items
        document.addEventListener('keydown', (e) => {
            const imageItem = e.target.closest('.image-item');
            if (imageItem && (e.key === 'Enter' || e.key === ' ')) {
                e.preventDefault();
                this.openLightbox(imageItem);
            }
        });
        
        // Close lightbox
        const closeLightbox = () => this.closeLightbox();
        closeBtn.addEventListener('click', closeLightbox);
        backdrop.addEventListener('click', closeLightbox);
        
        // Navigation
        prevBtn.addEventListener('click', () => this.navigateImage(-1));
        nextBtn.addEventListener('click', () => this.navigateImage(1));
        
        // Keyboard controls
        document.addEventListener('keydown', (e) => {
            if (!this.lightboxOpen) return;
            
            switch (e.key) {
                case 'Escape':
                    e.preventDefault();
                    this.closeLightbox();
                    break;
                case 'ArrowLeft':
                    e.preventDefault();
                    this.navigateImage(-1);
                    break;
                case 'ArrowRight':
                    e.preventDefault();
                    this.navigateImage(1);
                    break;
            }
        });
        
        // Prevent scrolling when lightbox is open
        lightbox.addEventListener('wheel', (e) => {
            if (this.lightboxOpen) {
                e.preventDefault();
            }
        }, { passive: false });
    }
    
    openLightbox(imageItem) {
        try {
            const images = JSON.parse(imageItem.dataset.images);
            const index = parseInt(imageItem.dataset.index);
            const date = imageItem.dataset.date;
            
            this.currentLightboxImages = images;
            this.currentImageIndex = index;
            this.lightboxOpen = true;
            
            // Update lightbox content
            this.elements.lightboxTitle.textContent = `${this.formatDate(date)} - Gallery`;
            this.updateLightboxImage();
            
            // Show lightbox
            this.elements.lightbox.classList.remove('hidden');
            this.elements.lightbox.classList.add('active');
            
            // Focus management
            this.elements.lightbox.focus();
            document.body.style.overflow = 'hidden';
            
            // Trap focus within lightbox
            this.trapFocus(this.elements.lightbox);
            
        } catch (error) {
            console.error('Error opening lightbox:', error);
        }
    }
    
    closeLightbox() {
        this.lightboxOpen = false;
        
        // Stop and reset video if playing
        const videoElement = this.elements.lightboxVideo;
        if (videoElement) {
            videoElement.pause();
            videoElement.currentTime = 0; // Reset to beginning
            videoElement.src = ''; // Clear source to free resources
        }
        
        this.elements.lightbox.classList.remove('active');
        
        // Small delay to allow animation to complete
        setTimeout(() => {
            this.elements.lightbox.classList.add('hidden');
        }, 220);
        
        // Restore body scroll and focus
        document.body.style.overflow = '';
        
        // Return focus to the image that opened the lightbox
        const currentImageItem = document.querySelector(`[data-index="${this.currentImageIndex}"]`);
        if (currentImageItem) {
            currentImageItem.focus();
        }
    }
    
    navigateImage(direction) {
        const newIndex = this.currentImageIndex + direction;
        
        if (newIndex >= 0 && newIndex < this.currentLightboxImages.length) {
            this.currentImageIndex = newIndex;
            this.updateLightboxImage();
        }
    }
    
    updateLightboxImage() {
        const currentMedia = this.currentLightboxImages[this.currentImageIndex];
        const imageElement = this.elements.lightboxImage;
        const videoElement = this.elements.lightboxVideo;
        const counter = this.elements.lightboxCounter;
        const isVideoFile = this.isVideo(currentMedia);
        
        // Update counter
        counter.textContent = `${this.currentImageIndex + 1} of ${this.currentLightboxImages.length}`;
        
        if (isVideoFile) {
            // Pause any currently playing video first
            if (videoElement && !videoElement.paused) {
                videoElement.pause();
                videoElement.currentTime = 0;
            }
            
            // Hide image and loading spinner, show video
            imageElement.style.display = 'none';
            const loadingSpinner = this.elements.lightbox.querySelector('.image-loading');
            if (loadingSpinner) {
                loadingSpinner.style.opacity = '0';
            }
            videoElement.style.display = 'block';
            videoElement.src = currentMedia;
            videoElement.muted = false; // Unmute in lightbox
            
            // Hide loading spinner when video metadata is loaded
            const hideLoading = () => {
                if (loadingSpinner) {
                    loadingSpinner.style.opacity = '0';
                }
            };
            
            videoElement.addEventListener('loadedmetadata', hideLoading, { once: true });
            videoElement.addEventListener('canplay', hideLoading, { once: true });
            
            videoElement.load();
            // Try to play the video
            videoElement.play().catch(err => {
                console.log('Video autoplay prevented:', err);
            });
        } else {
            // Hide video, show image - pause and reset video
            if (videoElement) {
                videoElement.style.display = 'none';
                videoElement.pause();
                videoElement.currentTime = 0;
                videoElement.src = '';
            }
            imageElement.style.display = 'block';
            
            // Show loading spinner for images
            const loadingSpinner = this.elements.lightbox.querySelector('.image-loading');
            if (loadingSpinner) {
                loadingSpinner.style.opacity = '1';
            }
            
            // Update image with loading state
            imageElement.style.opacity = '0';
            
            const newImage = new Image();
            newImage.onload = () => {
                imageElement.src = newImage.src;
                imageElement.alt = this.generateAltText(currentMedia, '', this.currentImageIndex);
                imageElement.style.opacity = '1';
                // Hide loading spinner when image loads
                if (loadingSpinner) {
                    loadingSpinner.style.opacity = '0';
                }
            };
            
            newImage.onerror = () => {
                console.error('Failed to load image:', currentMedia);
                imageElement.alt = 'Failed to load image';
                imageElement.style.opacity = '1';
                // Hide loading spinner even on error
                if (loadingSpinner) {
                    loadingSpinner.style.opacity = '0';
                }
            };
            
            newImage.src = currentMedia;
        }
        
        // Update navigation button states
        const prevBtn = this.elements.lightbox.querySelector('.lightbox-prev');
        const nextBtn = this.elements.lightbox.querySelector('.lightbox-next');
        
        prevBtn.style.opacity = this.currentImageIndex === 0 ? '0.5' : '1';
        nextBtn.style.opacity = this.currentImageIndex === this.currentLightboxImages.length - 1 ? '0.5' : '1';
    }
    
    trapFocus(element) {
        const focusableElements = element.querySelectorAll(
            'button, [href], input, select, textarea, [tabindex]:not([tabindex="-1"])'
        );
        
        const firstFocusableElement = focusableElements[0];
        const lastFocusableElement = focusableElements[focusableElements.length - 1];
        
        element.addEventListener('keydown', (e) => {
            if (e.key === 'Tab') {
                if (e.shiftKey) {
                    if (document.activeElement === firstFocusableElement) {
                        lastFocusableElement.focus();
                        e.preventDefault();
                    }
                } else {
                    if (document.activeElement === lastFocusableElement) {
                        firstFocusableElement.focus();
                        e.preventDefault();
                    }
                }
            }
        });
    }
    
    initNavigation() {
        // Smooth scroll for navigation links
        const navLinks = document.querySelectorAll('.nav-link[href^="#"]');
        
        navLinks.forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const targetId = link.getAttribute('href');
                const targetElement = document.querySelector(targetId);
                
                if (targetElement) {
                    const headerHeight = document.querySelector('.header').offsetHeight;
                    const targetPosition = targetElement.offsetTop - headerHeight - 20;
                    
                    window.scrollTo({
                        top: targetPosition,
                        behavior: 'smooth'
                    });
                    
                    // Update focus for accessibility
                    targetElement.setAttribute('tabindex', '-1');
                    targetElement.focus();
                    targetElement.addEventListener('blur', () => {
                        targetElement.removeAttribute('tabindex');
                    }, { once: true });
                }
            });
        });
        
        // Update active navigation state on scroll
        let ticking = false;
        
        const updateActiveNav = () => {
            const sections = document.querySelectorAll('section[id]');
            const scrollPos = window.scrollY + document.querySelector('.header').offsetHeight + 50;
            
            let activeSection = '';
            sections.forEach(section => {
                if (section.offsetTop <= scrollPos) {
                    activeSection = section.getAttribute('id');
                }
            });
            
            // Update navigation active states
            navLinks.forEach(link => {
                const targetId = link.getAttribute('href').substring(1);
                if (targetId === activeSection) {
                    link.style.color = 'var(--accent)';
                } else {
                    link.style.color = 'var(--muted)';
                }
            });
            
            ticking = false;
        };
        
        const requestUpdateActiveNav = () => {
            if (!ticking) {
                requestAnimationFrame(updateActiveNav);
                ticking = true;
            }
        };
        
        window.addEventListener('scroll', requestUpdateActiveNav);
        window.addEventListener('resize', requestUpdateActiveNav);
        
        // Initial call
        updateActiveNav();
    }
    
    // ===== SCHEDULE FUNCTIONALITY =====
    
    renderSchedule() {
        if (!this.elements.scheduleTbody) {
            console.log('Schedule section not available');
            return;
        }
        
        if (!this.scheduleData || this.scheduleData.schedule.length === 0) {
            this.elements.scheduleTbody.innerHTML = `
                <tr>
                    <td colspan="4" style="text-align: center; padding: var(--space-2xl); color: var(--muted);">
                        لا توجد بيانات جدولة متاحة حالياً
                    </td>
                </tr>
            `;
            return;
        }
        
        this.updateScheduleTable();
    }
    
    updateScheduleTable() {
        // Display project dates
        if (this.scheduleData && this.scheduleData.project) {
            const projectStart = this.scheduleData.project.start_date;
            const projectEnd = this.scheduleData.project.end_date;
            this.renderProjectDates(projectStart, projectEnd);
        }
        
        const rows = this.filteredScheduleData.map(item => {
            const work = item.work || '—';
            const duration = item.duration_text || '—';
            const completionPercentage = item.completion_percentage || 0;
            
            // Format progress display
            const progressDisplay = completionPercentage > 0 ? `${completionPercentage}%` : '—';
            
            // Determine status based on individual task progress percentage
            const taskStatus = this.getStatusFromProgress(completionPercentage);
            
            return `
                <tr role="row">
                    <td class="work-cell" data-label="العمل">
                        <span class="cell-content">${this.escapeHtml(work)}</span>
                    </td>
                    <td class="duration-cell" data-label="مدة العمل">
                        <span class="cell-content">${this.escapeHtml(duration)}</span>
                    </td>
                    <td class="notes-cell" data-label="نسبه الانجاز">
                        <span class="cell-content">${this.escapeHtml(progressDisplay)}</span>
                    </td>
                    <td class="status-cell" data-label="الحالة اليوم">
                        <span class="cell-content">${this.getOverallStatusPill(taskStatus)}</span>
                    </td>
                </tr>
            `;
        }).join('');
        this.elements.scheduleTbody.innerHTML = rows;
    }
    
    renderProjectDates(startDate, endDate) {
        const datesElement = document.getElementById('project-dates');
        if (!datesElement) return;
        
        if (startDate && endDate) {
            const formattedStart = this.formatDateForDisplay(startDate);
            const formattedEnd = this.formatDateForDisplay(endDate);
            
            datesElement.innerHTML = `
                <div class="project-dates-info">
                    <span class="date-label">تاريخ البدء:</span>
                    <span class="date-value">${this.escapeHtml(formattedStart)}</span>
                    <span class="date-separator">•</span>
                    <span class="date-label">تاريخ التسليم:</span>
                    <span class="date-value">${this.escapeHtml(formattedEnd)}</span>
                </div>
            `;
        } else {
            datesElement.innerHTML = '';
        }
    }
    
    formatDateForDisplay(dateString) {
        // Format D/M/YYYY to a more readable format
        const parts = dateString.split('/');
        if (parts.length === 3) {
            const day = parts[0];
            const month = parts[1];
            const year = parts[2];
            return `${day}/${month}/${year}`;
        }
        return dateString;
    }
    
    getStatusFromProgress(progressPercent) {
        if (progressPercent === null || progressPercent === undefined) {
            return "Not Started";
        }
        
        if (progressPercent >= 100) {
            return "Completed";
        } else if (progressPercent > 0) {
            return "In Progress";
        } else {
            return "Not Started";
        }
    }
    
    getOverallStatusPill(status) {
        if (!status) {
            return '<span class="status-pill status-not-started">—</span>';
        }
        
        switch (status) {
            case "Not Started":
                return '<span class="status-pill status-not-started">لم يبدأ</span>';
            case "In Progress":
                return '<span class="status-pill status-on-time">جاري العمل</span>';
            case "Completed":
                return '<span class="status-pill status-completed">مكتمل</span>';
            default:
                return '<span class="status-pill status-not-started">—</span>';
        }
    }
    
    initSchedule() {
        if (!this.elements.scheduleTbody) {
            return;
        }
        
        // Sort buttons
        const sortButtons = {
            'sort-start-asc': () => this.sortSchedule('start_date', 'asc'),
            'sort-start-desc': () => this.sortSchedule('start_date', 'desc'),
            'sort-end-asc': () => this.sortSchedule('end_date', 'asc'),
            'sort-end-desc': () => this.sortSchedule('end_date', 'desc')
        };
        
        Object.entries(sortButtons).forEach(([id, handler]) => {
            const button = document.getElementById(id);
            if (button) {
                button.addEventListener('click', () => {
                    // Update active state
                    document.querySelectorAll('.sort-btn').forEach(btn => btn.classList.remove('active'));
                    button.classList.add('active');
                    handler();
                });
            }
        });
        
        // Search functionality
        if (this.elements.scheduleSearch) {
            this.elements.scheduleSearch.addEventListener('input', (e) => {
                this.filterSchedule(e.target.value);
            });
        }
    }
    
    sortSchedule(field, direction) {
        this.currentSortField = field;
        this.currentSortDirection = direction;
        
        this.filteredScheduleData.sort((a, b) => {
            let aValue = a[field];
            let bValue = b[field];
            
            // Try to parse dates
            const aDate = this.parseDate(aValue);
            const bDate = this.parseDate(bValue);
            
            if (aDate && bDate) {
                // Both are valid dates
                aValue = aDate;
                bValue = bDate;
            } else {
                // Fallback to string comparison
                aValue = aValue ? aValue.toString() : '';
                bValue = bValue ? bValue.toString() : '';
            }
            
            let comparison = 0;
            if (aValue < bValue) {
                comparison = -1;
            } else if (aValue > bValue) {
                comparison = 1;
            }
            
            return direction === 'desc' ? -comparison : comparison;
        });
        
        this.updateScheduleTable();
    }
    
    filterSchedule(searchTerm) {
        const term = searchTerm.toLowerCase().trim();
        
        if (!term) {
            this.filteredScheduleData = [...this.scheduleData.schedule];
        } else {
            this.filteredScheduleData = this.scheduleData.schedule.filter(item => 
                (item.work && item.work.toLowerCase().includes(term))
            );
        }
        
        // Re-apply current sort if any
        if (this.currentSortField) {
            this.sortSchedule(this.currentSortField, this.currentSortDirection);
        } else {
            this.updateScheduleTable();
        }
    }
    
    parseDate(dateString) {
        if (!dateString || dateString === '—') {
            return null;
        }
        
        // Parse D/M/YYYY format
        const parts = dateString.split('/');
        if (parts.length === 3) {
            const day = parseInt(parts[0], 10);
            const month = parseInt(parts[1], 10) - 1; // JS months are 0-indexed
            const year = parseInt(parts[2], 10);
            
            if (!isNaN(day) && !isNaN(month) && !isNaN(year)) {
                const date = new Date(year, month, day);
                // Validate the date
                if (date.getDate() === day && date.getMonth() === month && date.getFullYear() === year) {
                    return date;
                }
            }
        }
        
        return null;
    }

    calculateOverallProgress(startDateStr, endDateStr, todayDateStr = null) {
        // Parse dates
        const startDate = this.parseDate(startDateStr);
        const endDate = this.parseDate(endDateStr);
        const today = todayDateStr ? this.parseDate(todayDateStr) : new Date();
        
        // Reset time to midnight for accurate day calculations
        if (startDate) {
            startDate.setHours(0, 0, 0, 0);
        }
        if (endDate) {
            endDate.setHours(0, 0, 0, 0);
        }
        if (today) {
            today.setHours(0, 0, 0, 0);
        }
        
        // Validate dates
        if (!startDate || !endDate) {
            return null;
        }
        
        // Calculate total days (inclusive): from start_date to end_date
        const totalDaysInclusive = Math.floor((endDate - startDate) / (1000 * 60 * 60 * 24)) + 1;
        
        // Calculate elapsed days (inclusive): from start_date to today
        let elapsedDaysInclusive = 0;
        let status = "Not Started";
        
        if (today < startDate) {
            // Before start date
            elapsedDaysInclusive = 0;
            status = "Not Started";
        } else if (today > endDate) {
            // After end date
            elapsedDaysInclusive = totalDaysInclusive;
            status = "Completed";
        } else {
            // Between start and end dates (inclusive)
            elapsedDaysInclusive = Math.floor((today - startDate) / (1000 * 60 * 60 * 24)) + 1;
            status = "In Progress";
        }
        
        // Calculate overall progress percent
        const overallProgressPercent = totalDaysInclusive > 0 
            ? Math.round((elapsedDaysInclusive / totalDaysInclusive) * 100 * 100) / 100 
            : 0;
        
        // Return JSON object
        return {
            "start_date": startDateStr,
            "end_date": endDateStr,
            "total_days_inclusive": totalDaysInclusive,
            "elapsed_days_inclusive": elapsedDaysInclusive,
            "overall_progress_percent": overallProgressPercent,
            "status": status
        };
    }

    getSmartStatusPill(item) {
        const now = new Date();
        const startDate = this.parseDate(item.start_date);
        const endDate = this.parseDate(item.end_date);
        const completionPercentage = parseInt(item.completion_percentage) || 0;

        // If no valid dates, check completion percentage only
        if (!startDate || !endDate) {
            if (completionPercentage >= 100) {
                return '<span class="status-pill status-completed">مكتمل</span>';
            } else if (completionPercentage > 0) {
                return '<span class="status-pill status-on-time">جاري العمل</span>';
            } else {
                return '<span class="status-pill status-not-started">لم يبدأ</span>';
            }
        }

        // If work is completed (100%)
        if (completionPercentage >= 100) {
            return '<span class="status-pill status-completed">مكتمل</span>';
        }

        // If current date is before start date
        if (now < startDate) {
            if (completionPercentage > 0) {
                return '<span class="status-pill status-on-time">بدء مبكر</span>';
            } else {
                return '<span class="status-pill status-not-started">لم يبدأ</span>';
            }
        }

        // Calculate progress expectations based on time elapsed
        const totalDuration = endDate - startDate;
        const elapsedDuration = now - startDate;
        const expectedProgress = Math.max(0, Math.min(100, (elapsedDuration / totalDuration) * 100));

        // Calculate days remaining
        const daysRemaining = Math.ceil((endDate - now) / (1000 * 60 * 60 * 24));

        // If current date is after end date (overdue)
        if (now > endDate) {
            if (completionPercentage >= 95) {
                return '<span class="status-pill status-warning">شبه مكتمل</span>';
            } else {
                return '<span class="status-pill status-delayed">متأخر</span>';
            }
        }

        // Critical status: Close to deadline with low completion
        if (daysRemaining <= 2 && completionPercentage < 80) {
            return '<span class="status-pill status-critical">حرج</span>';
        }

        // Warning status: Behind expected progress
        if (completionPercentage < expectedProgress - 15) {
            return '<span class="status-pill status-warning">تحذير</span>';
        }

        // Critical if very behind schedule
        if (completionPercentage < expectedProgress - 30) {
            return '<span class="status-pill status-critical">حرج</span>';
        }

        // On time if progress matches or exceeds expectation
        if (completionPercentage >= expectedProgress - 5) {
            return '<span class="status-pill status-on-time">في الموعد</span>';
        }

        // Default to warning for slightly behind
        return '<span class="status-pill status-warning">تحذير</span>';
    }
}

// Initialize the application when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        new HouseProgressTracker();
    });
} else {
    new HouseProgressTracker();
}

// Export for potential debugging
window.HouseProgressTracker = HouseProgressTracker;
        